/* $Id: siclimap.c,v 1.8 1998/07/30 23:22:40 ericb Exp $ */
/* Copyright (C) 1997 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Deals with mapping memory for SICL.  This is surprisingly
   complicated, due in part to misfeatures/bugs in SICL and in part to
   our need to map in so much memory.

   Constraints we need to deal with:

   a. We must always have an A16 window around, so that we can ensure
      quick access to any module's A16 registers.  This will help make
      commands faster (since they use only A16 registers), and will
      make data handshake faster once I move the handshake register to
      A16.

   b. SICL will gratuitously refuse to map more than five windows
      associated with a single session id.  To go beyond five, it is
      necessary to open multiple sessions.  This is true on all
      versions of SICL and all platforms that I've tried it on -
      C_03_08a on HP-UX 9.05, E.01.01 on HP-UX 10.20, V743, MXI, and
      so on.  When this limit of five is reached, the imap call fails
      with I_ERR_BADMAP.  So we will need to keep a list of session
      ids around, and when doing imap we will have to go through the
      list of ids looking for one that works for us.  Ick.

   c. There is a limit to the total number of maps that can be
      simultaneously open, even if multiple sessions are used.  This
      limit is apparently determined by hardware.  When using MXI on
      an external host computer, there are only 10 1MB windows and 16
      64KB windows available.  On a V743, the limit is roughly 53
      total windows of any size.  If this limit is reached and another
      imap is done, and the lockwait flag is set (it is by default)
      the imap will hang for whatever timeout has been set (default
      forever), and then return I_ERR_TIMEOUT.  Because of this, it is
      important to clear the lockwait flag using isetlockwait.

   d. When using an external host computer with MXI, all accesses
      become D16 accesses, even if explicitly requested as D32 with
      ilblockcopy or ilpeek.  The only way to get around this is to
      map the memory using I_MAP_A24_D32.  If I_MAP_A24_D32 is used,
      then all accesses become D32 accesses, even if explicitly
      requested as D16.  So we must try to use I_MAP_A24_D32 to get
      the best transfer rates, but we can't use it when using local
      bus, because then we must always use D16 accesses.
      Additionally, I_MAP_A24_D32 is always rejected when on a V743,
      so we must allow for this error and fall back on using
      I_MAP_A24.  This is an issue only when mapping in the E143x
      fifos, since all other accesses can be D16 without hurting much.

   e. SICL has a bug when using a device session and I_MAP_A24_D32.
      This bug causes the imap to permanently lock the memory and
      iunmap fails.  The memory can't be recovered until the unix
      system is rebooted.  Because of this bug, we avoid device
      sessions entirely.  This bug is in C_03_08a and E.01.01.
      Additionally, on C_03_08 a device session imap of I_MAP_A24_D32
      will appear to work but in fact it does not work at all.

   f. Because of the need to use I_MAP_A24_D32 in some cases but not
      others, we map the E143x data fifo separately from the rest of
      the A24 memory.  This means each module uses two A24 memory
      maps.  We must handle large multi-mainframe systems with up to
      32 (or maybe even 64) E143x modules.  This means we can't have
      all those maps open at once, so we will have to close and open
      maps on the fly.  However, this is probably not very fast, so we
      want to avoid it when the module count is small.

      If we are openning and closing memory on the fly, we probably
      should close the most recently used memory, because the modules
      will normally be cycled through, and closing the least recently
      used would probably result in constant thrashing.

   g. SICL also has a bug when you open and close memory maps in the
      wrong way.  When you do this, unmapping one map may cause other
      maps to stop working.  I am attempting to work around this bug
      by always unmapping in the reverse order that the maps were
      originally in.  We'll see if that works.  If not, we may just be
      hosed. */

#include "sema.h"

#ifdef	HAVE_SICL

/* #define	DEBUG_SICL */		/* For low-level SICL calls */
/* #define	DEBUG_LIST */		/* For mid-level iopen list calls */
/* #define	DEBUG_I1432_SICL */	/* For high-level i1432_sicl calls */

/*
 *********************************************************************
 Wrappers for iopen, iclose, imap, and iunmap.  These allow for
 easier debugging, and deal with timeout and lockwait issues.  They
 have no knowledge of the E1432 library.
 *********************************************************************
 */

static INST
my_iopen(char *addr)
{
    INST    rv;

#ifdef	DEBUG_SICL
    (void) printf("iopen(\"%s\"): ", addr);
#ifndef	E1485_SOURCE
    (void) fflush(stdout);
#endif
#endif

    rv = iopen(addr);

#ifdef	DEBUG_SICL
    if (rv == 0)
	(void) printf("%s\n", igeterrstr(igeterrno()));
    else
	(void) printf("%d\n", rv);
#endif

#ifndef EMULATE_SICL
    /* This function doesn't exist in esicl.  But presumably we only
       talk to a few modules at a time with esicl, so we'll never run
       into a case where an imap would block. */
    (void) isetlockwait(rv, 0);
#endif
    (void) itimeout(rv, E1432_SICL_TIMEOUT);

    return rv;
}

static int
my_iclose(INST id)
{
    int     rv;

#ifdef	DEBUG_SICL
    (void) printf("iclose(%d): ", id);
#ifndef	E1485_SOURCE
    (void) fflush(stdout);
#endif
#endif

    rv = iclose(id);

#ifdef	DEBUG_SICL
    if (rv != 0)
	(void) printf("returned %d: %s\n", rv, igeterrstr(rv));
    else
	(void) printf("successful\n");
#endif

    return rv;
}

static char *
my_imap(INST id, int map_space, unsigned int pagestart,
	unsigned int pagecnt, char *suggested)
{
    char   *rv;

#ifdef	DEBUG_SICL
    (void) printf("imap(%d, %d, %u, %u, 0x%p): ",
		  id, map_space, pagestart, pagecnt, suggested);
#ifndef	E1485_SOURCE
    (void) fflush(stdout);
#endif
#endif

    rv = imap(id, map_space, pagestart, pagecnt, suggested);

#ifdef	DEBUG_SICL
    if (rv == NULL)
	(void) printf("%s\n", igeterrstr(igeterrno()));
    else
	(void) printf("0x%p\n", rv);
#endif

    return rv;
}

static int
my_iunmap(INST id, char *addr, int map_space, unsigned int pagestart,
	  unsigned int pagecnt)
{
    int     rv;

#ifdef	DEBUG_SICL
    (void) printf("iunmap(%d, 0x%p, %d, %u, %u): ",
		  id, addr, map_space, pagestart, pagecnt);
#ifndef	E1485_SOURCE
    (void) fflush(stdout);
#endif
#endif

    rv = iunmap(id, addr, map_space, pagestart, pagecnt);

#ifdef	DEBUG_SICL
    if (rv != 0)
	(void) printf("returned %d: %s\n", rv, igeterrstr(rv));
    else
	(void) printf("successful\n");
#endif

    return rv;
}

/*
 *********************************************************************
 Functions to deal with the list of SICL sessions we must keep
 around.  These functions do all of the iopen, iclose, imap, and
 iunmap calls for the E1432 library.  These functions have no
 knowledge of the E1432 library.

 For the moment, our strategy with iopen is to maintain an array of
 ids, with a count specifying how many ids we have.  We make no
 attempt to close ids in the middle of the array, so there is no
 need to search for an unused array entry to use.
 *********************************************************************
 */

/* Initialize the iopen list.  If RESET is non-zero, assume the list
   was already in use and close any open sessions */
static int
iopen_list_init(struct i1432_iopen_list_info *info, int reset)
{
    int     i;

#ifdef	DEBUG_LIST
    (void) printf("iopen_list_init(%d)\n", reset);
#endif

    /* Clear id list counter */
    info->nid = 0;

    /* Clear id array.  Do last in case my_iclose errors. */
    for (i = 0; i < E1432_SICL_ID_MAX; i++)
    {
	if (reset && info->id[i] != 0)
	    if (my_iclose(info->id[i]) != 0)
		return -1;
	info->id[i] = 0;
    }

    return 0;
}

/* Open a new SICL session and put it at the end of the list */
static int
iopen_list_add(struct i1432_iopen_list_info *info)
{
    /* Don't overrun the array of ids */
    if (info->nid >= E1432_SICL_ID_MAX)
	return -1;

    /* Do the iopen */
    info->id[info->nid] = my_iopen(info->name);
    if (info->id[info->nid] == 0)
	return -1;

    /* Mark as available for imap */
    info->avail[info->nid] = 1;

#ifdef	DEBUG_LIST
    (void) printf("iopen_list_add: id[%d] = %d, avail[%d] = 1\n",
		  info->nid, info->id[info->nid], info->nid);
#endif

    /* Increment id list counter */
    info->nid++;

    return 0;
}

/* Close the most recently openned sicl session */
static int
iopen_list_remove(struct i1432_iopen_list_info *info)
{
    /* Don't underrun the array of ids */
    if (info->nid <= 0 || info->nid > E1432_SICL_ID_MAX)
	return -1;

    /* Decrement id list counter */
    info->nid--;

    /* If already closed, error */
    if (info->id[info->nid] == 0)
	return -1;

    /* Do the iclose */
    if (my_iclose(info->id[info->nid]) != 0)
	return -1;

    /* Mark as closed */
    info->id[info->nid] = 0;

#ifdef	DEBUG_LIST
    (void) printf("iopen_list_remove: id[%d]\n", info->nid);
#endif

    return 0;
}

/* Map in some VXI memory.  This will open a new SICL session if
   necessary. */
static char *
iopen_list_imap(struct i1432_iopen_list_info *info, int *id_index,
		int map_space, unsigned int pagestart,
		unsigned int pagecnt, char *suggested)
{
    int     i, error;
    char   *addr;

    /* Find a SICL session to use */
    for (i = 0; i < info->nid; i++)
    {
	/* Don't bother trying if we know it will fail */
	if (!info->avail[i])
	    continue;

	/* Try the imap */
	addr = my_imap(info->id[i], map_space, pagestart, pagecnt, suggested);
	if (addr != 0)
	{
	    /* Success */
#ifdef	DEBUG_LIST
	    (void) printf("iopen_list_imap: used id[%d] = %d\n",
			  i, info->id[i]);
#endif
	    *id_index = i;
	    return addr;
	}

	/* This imap attempt failed, mark the session as not available */
	info->avail[i] = 0;

#ifdef	DEBUG_LIST
	(void) printf("iopen_list_imap: avail[%d] = 0\n", i);
#endif
    }

    /* All existing SICL sessions failed, so open a new SICL session */
    error = iopen_list_add(info);
    if (error < 0)
	return NULL;

    /* Try the imap using the new session */
    addr = my_imap(info->id[info->nid - 1], map_space, pagestart,
		   pagecnt, suggested);
    if (addr != 0)
    {
	/* Success */
#ifdef	DEBUG_LIST
	(void) printf("iopen_list_imap: used id[%d] = %d\n",
		      info->nid - 1, info->id[info->nid - 1]);
#endif
	*id_index = info->nid - 1;
	return addr;
    }

    /* The imap still failed, close the new SICL session and give up */
    (void) iopen_list_remove(info);
    return NULL;
}

/* Unmap some VXI memory that was previously mapped */
static int
iopen_list_iunmap(struct i1432_iopen_list_info *info, int id_index,
		  char *addr, int map_space,
		  unsigned int pagestart, unsigned int pagecnt)
{
    /* Do the iunmap */
    if (my_iunmap(info->id[id_index], addr, map_space,
		  pagestart, pagecnt) != 0)
	return -1;

    /* Mark this session as available for more imaps */
    info->avail[id_index] = 1;

#ifdef	DEBUG_LIST
    (void) printf("iopen_list_iunmap: avail[%d] = 1\n", id_index);
#endif

    return 0;
}

/*
 *********************************************************************
 These functions are called by the rest of the E1432 library.
 *********************************************************************
 */

SHORTSIZ16
i1432_sicl_imap_init(void)
{
#if defined(I_MAP_A24_D32) && !defined(NO_D32)
    int     dummy, status;
#endif

    i1432_sicl_imap_count = 0;

    /* Initialize iopen list */
    i1432_iopen_list_info.name = i1432_sicl_name;
    if (iopen_list_init(&i1432_iopen_list_info, 0) < 0)
	return i1432_print_error(ERR1432_SICL_ERROR);

    /* Map in A16 memory */
    i1432_a16_base = (SHORTSIZ16 *) (void *)
	iopen_list_imap(&i1432_iopen_list_info, &i1432_a16_id_index,
			I_MAP_A16, 0, 0, NULL);
    if (i1432_a16_base == NULL)
    {
	DIAG_PRINTF(1, ("  VXI interface failure.\n"));
	DIAG_PRINTF(1, ("  This is not likely a E1432 problem.\n"));
	DIAG_PRINTF(1, ("  The SICL call to imap(A16) errored:  %s \n",
			igeterrstr(igeterrno()) ));
	i1432_error_info = igeterrstr(igeterrno());
	return i1432_print_error(ERR1432_UNABLE_TO_MAP_MEMORY);
    }

    i1432_sicl_imap_count++;

    /* Save the SICL id */
    i1432_sicl_id = i1432_iopen_list_info.id[i1432_a16_id_index];

    /* Test for ability to use I_MAP_A24_D32 */
#if defined(I_MAP_A24_D32) && !defined(NO_D32)
    dummy = 0;	/* Attempt to keep Purify happy */
    status = imapinfo(i1432_sicl_id, I_MAP_A24_D32, &dummy, &dummy);
    i1432_have_a24_d32 = (status == 0);
#else
    i1432_have_a24_d32 = 0;
#endif

    if (i1432_have_a24_d32)
	/* This is probably an external controller using a MXI
	   connection to the system.  If so, we will be able to do
	   approximately 19 maps before SICL runs out.  Seems like it
	   should be 26 (10 1MB windows + 16 64K windows), but in
	   practice I've been limited to 19.  We want to be a nice VXI
	   citizen and leave something for non-E143x use, so I'll
	   limit our use to 15 of those 19.  That means we won't be
	   unmapping unless there are eight or more E143x modules. */
	i1432_sicl_imap_max = 15;
    else
	/* This is probably an embedded controller with lots of
	   windows.  We still want to be a nice VXI citizen, so I'll
	   limit our use to 33.  We won't be unmapping unless there
	   are 17 or more E143x modules. */
	i1432_sicl_imap_max = 33;

#ifdef	DEBUG_I1432_SICL
    (void) printf("i1432_sicl_imap_init: A16 id %d, A24_D32 %d, imap max %d\n",
		  i1432_sicl_id, i1432_have_a24_d32,
		  i1432_sicl_imap_max);
#endif

    return 0;
}

SHORTSIZ16
i1432_sicl_imap_uninit(void)
{
#ifdef	DEBUG_I1432_SICL
    (void) printf("i1432_sicl_imap_uninit\n");
#endif

    if (i1432_a16_base != NULL)
    {
	if (iopen_list_iunmap(&i1432_iopen_list_info,
			      i1432_a16_id_index,
			      (char *) i1432_a16_base,
			      I_MAP_A16, 0, 0) < 0)
	    return i1432_print_error(ERR1432_SICL_ERROR);
	i1432_a16_base = NULL;

	i1432_sicl_imap_count--;
    }

    if (iopen_list_init(&i1432_iopen_list_info, 1) < 0)
	return i1432_print_error(ERR1432_SICL_ERROR);

    if (i1432_sicl_imap_count != 0)
    {
	i1432_sicl_imap_count = 0;
	return i1432_print_error(ERR1432_LIST_ERROR);
    }

    return 0;
}

SHORTSIZ16
i1432_sicl_imap_setup(E1432_MODULE_LIST_NODE *mn, int big_map)
{
    struct vxiinfo info;
    int     ierror;

    mn->a24_fifo_id_index = -1;
    mn->a24_imap_count = -1;
    mn->a24_fifo_imap_count = -1;

    /* Get first A24 page on this module */
    ierror = ivxirminfo(i1432_sicl_id, mn->la, &info);
    if (ierror)
    {
	i1432_error_info = igeterrstr(ierror);
	return i1432_la_print_error(mn->la, ERR1432_SICL_ERROR);
    }
    /* Convert to 64K pages */
    mn->a24_pagestart = (unsigned int) info.memstart >> 8;
    if (mn->a24_256k)
	mn->a24_pagestart += 2;	/* Use movable window, it's bigger */

    /* Calculate how many (64K) pages to imap */
    if (big_map && !mn->a24_256k)
	mn->a24_pagecnt = 16;
    else
	mn->a24_pagecnt = 2;

    /* Calculate the starting location of the FIFO */
    if (mn->a24_256k)
	mn->a24_fifo_pagestart = (unsigned int) (FIFO_PAGE_START_256K - 2);
    else
	mn->a24_fifo_pagestart = FIFO_PAGE_START_1M;

    /* Set this to I_MAP_A24 for now.  i1432_sicl_imap_setup2 may
       later set it to I_MAP_A24_D32 if appropriate. */
    mn->a24_fifo_map_space = I_MAP_A24;

    return 0;
}

/* Originally this was part of i1432_sicl_imap_setup.  But that causes
   problems when doing eavesdrop monitoring of thruput.
   i1432_sicl_imap_setup is called during e1432_assign_channels,
   before we know whether thruput is even going to happen.  So the
   code at that time decided that D32 was OK, and set
   mn->a24_fifo_map_space to I_MAP_A24_D32.  That corrupts the thruput
   data if monitoring is done.

   Instead, this must be called at the start of the measurement, when
   we know for sure whether thruput is taking place. */
SHORTSIZ16
i1432_sicl_imap_setup2(E1432_MODULE_LIST_NODE *mn)
{
#if defined(I_MAP_A24_D32) && !defined(NO_D32)
    /* Strangely enough, if we successfully use I_MAP_A24_D32, then
       all accesses through this mapped memory end up as D32 accesses,
       even if we explicitly attempt to do D16 with iwblockcopy.  This
       is a problem if we are using the local bus, since any D32
       access can corrupt the local-bus data stream.  So, we can only
       ask for I_MAP_A24_D32 if we are not going to use the local bus.
       The test for local bus was already done in
       e1432_cached_parm_update, so we can just look at the d32 flag
       for this module. */
    if (mn->d32 && i1432_have_a24_d32)
    {
#ifdef	DEBUG_I1432_SICL
	(void) printf("i1432_sicl_imap_setup: la %d, A24_D32\n", mn->la);
#endif
	mn->a24_fifo_map_space = I_MAP_A24_D32;
    }
    else
#endif	/* I_MAP_A24_D32 && !NO_D32 */
    {
#ifdef	DEBUG_I1432_SICL
	(void) printf("i1432_sicl_imap_setup: la %d, A24\n", mn->la);
#endif
	mn->a24_fifo_map_space = I_MAP_A24;
    }

    return 0;
}

static SHORTSIZ16
sicl_iunmap_normal_help(E1432_MODULE_LIST_NODE *mn)
{
    if (mn->a24_base != NULL)
    {
#ifdef	DEBUG_I1432_SICL
	(void) printf("sicl_iunmap_normal_help: la %d\n", mn->la);
#endif

	if (iopen_list_iunmap(&i1432_iopen_list_info,
			      mn->a24_id_index,
			      (char *) mn->a24_base, I_MAP_A24,
			      mn->a24_pagestart, mn->a24_pagecnt) < 0)
	    return i1432_la_print_error(mn->la, ERR1432_SICL_ERROR);

	mn->a24_base = NULL;
	mn->a24_imap_count = -1;
	i1432_sicl_imap_count--;
    }

    return 0;
}

static SHORTSIZ16
sicl_iunmap_fifo_help(E1432_MODULE_LIST_NODE *mn)
{
    if (mn->a24_fifo_base != NULL)
    {
#ifdef	DEBUG_I1432_SICL
	(void) printf("sicl_iunmap_fifo_help: la %d\n", mn->la);
#endif

	if (iopen_list_iunmap(&i1432_iopen_list_info,
			      mn->a24_fifo_id_index,
			      (char *) mn->a24_fifo_base,
			      mn->a24_fifo_map_space,
			      mn->a24_pagestart + mn->a24_fifo_pagestart,
			      FIFO_PAGE_NUM) < 0)
	    return i1432_la_print_error(mn->la, ERR1432_SICL_ERROR);

	mn->a24_fifo_base = NULL;
	mn->a24_fifo_imap_count = -1;
	i1432_sicl_imap_count--;
    }

    return 0;
}

static SHORTSIZ16
sicl_iunmap_down_to(int imap_count)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;
    int     map, i;
    int     found;

#ifdef	DEBUG_I1432_SICL
    (void) printf("sicl_iunmap_down_to(%d) (current %d) (max %d)\n",
		  imap_count, i1432_sicl_imap_count,
		  i1432_sicl_imap_max);
#endif

    if (imap_count <= 0 || imap_count > i1432_sicl_imap_max)
    {
	/* Should not happen */
#ifdef	DEBUG_I1432_SICL
	(void) printf("sicl_iunmap_down_to: list error: %d\n", imap_count);
#endif
	return i1432_print_error(ERR1432_LIST_ERROR);
    }

    for (map = i1432_sicl_imap_count - 1; map >= imap_count; map--)
    {
	/* Search for imap number <map>, and unmap it */
#ifdef	DEBUG_I1432_SICL
	(void) printf("sicl_iunmap_down_to: searching for map %d\n", map);
#endif
	mn = i1432_mod_list;
	found = 0;
	for (i = 0; i < i1432_mod_count; i++, mn++)
	{
	    if (mn->a24_imap_count == map)
	    {
#ifdef	DEBUG_I1432_SICL
		(void) printf("sicl_iunmap_down_to: LA %d: normal unmap\n",
			      mn->la);
#endif
		error = sicl_iunmap_normal_help(mn);
		if (error)
		    return error;
		found = 1;
		break;
	    }
	    else if (mn->a24_fifo_imap_count == map)
	    {
#ifdef	DEBUG_I1432_SICL
		(void) printf("sicl_iunmap_down_to: LA %d: fifo unmap\n",
			      mn->la);
#endif
		error = sicl_iunmap_fifo_help(mn);
		if (error)
		    return error;
		found = 1;
		break;
	    }

#ifdef	DEBUG_I1432_SICL
	    (void) printf("sicl_iunmap_down_to: LA %d: skipped\n",
			  mn->la);
#endif

	    if (mn->a24_imap_count > map || mn->a24_fifo_imap_count > map)
	    {
		/* Should not happen */
#ifdef	DEBUG_I1432_SICL
		(void) printf("sicl_iunmap_down_to: LA %d: list error\n",
			      mn->la);
#endif
		return i1432_la_print_error(mn->la, ERR1432_LIST_ERROR);
	    }
	}
	if (!found)
	{
	    /* Should not happen */
#ifdef	DEBUG_I1432_SICL
	    (void) printf("sicl_iunmap_down_to: list error2\n");
#endif
	    return i1432_print_error(ERR1432_LIST_ERROR);
	}
    }

    return 0;
}

static SHORTSIZ16
sicl_imap_normal(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error;

    if (mn->a24_base != NULL)
	/* Already mapped, just use it */
	return 0;

#ifdef	DEBUG_I1432_SICL
    (void) printf("sicl_imap_normal: la %d\n", mn->la);
#endif

    /* Maybe unmap other memory, to make sure this map works */
    error = sicl_iunmap_down_to(i1432_sicl_imap_max - 1);
    if (error)
	return error;

    /* Do the imap */
    mn->a24_base = (LONGSIZ32 *) (void *)
	iopen_list_imap(&i1432_iopen_list_info, &mn->a24_id_index,
			I_MAP_A24, mn->a24_pagestart, mn->a24_pagecnt, NULL);
    if (mn->a24_base == NULL)
    {
#ifdef	DEBUG_I1432_SICL
	(void) printf("sicl_imap_normal: unmapping and retrying\n");
#endif

	/* Map failed.  Unmap something else and try again. */
	error = sicl_iunmap_down_to(i1432_sicl_imap_count - 1);
	if (error)
	    return error;

	/* Do the imap again */
	mn->a24_base = (LONGSIZ32 *) (void *)
	    iopen_list_imap(&i1432_iopen_list_info, &mn->a24_id_index,
			    I_MAP_A24, mn->a24_pagestart,
			    mn->a24_pagecnt, NULL);
	if (mn->a24_base == NULL)
	{
	    i1432_error_info = igeterrstr(igeterrno());
	    return i1432_la_print_error(mn->la,
					ERR1432_UNABLE_TO_MAP_MEMORY);
	}
    }

    /* Save the imap count, for use when unmapping */
    mn->a24_imap_count = i1432_sicl_imap_count++;

    /* Save the SICL id, for ilblockcopy */
    mn->sicl_id = i1432_iopen_list_info.id[mn->a24_id_index];

    return 0;
}

static SHORTSIZ16
sicl_imap_fifo(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error;

    if (mn->a24_fifo_base != NULL)
	/* Already mapped, just use it */
	return 0;

#ifdef	DEBUG_I1432_SICL
    (void) printf("sicl_imap_fifo: la %d\n", mn->la);
#endif

    /* Maybe unmap other memory, to make sure this map works */
    error = sicl_iunmap_down_to(i1432_sicl_imap_max - 1);
    if (error)
	return error;

    /* Do the imap */
    mn->a24_fifo_base = (LONGSIZ32 *) (void *)
	iopen_list_imap(&i1432_iopen_list_info, &mn->a24_fifo_id_index,
			mn->a24_fifo_map_space,
			mn->a24_pagestart + mn->a24_fifo_pagestart,
			FIFO_PAGE_NUM, NULL);
    if (mn->a24_fifo_base == NULL)
    {
#ifdef	DEBUG_I1432_SICL
	(void) printf("sicl_imap_fifo: unmapping and retrying\n");
#endif

	/* Map failed.  Unmap something else and try again. */
	error = sicl_iunmap_down_to(i1432_sicl_imap_count - 1);
	if (error)
	    return error;

	/* Do the imap again */
	mn->a24_fifo_base = (LONGSIZ32 *) (void *)
	    iopen_list_imap(&i1432_iopen_list_info, &mn->a24_fifo_id_index,
			    mn->a24_fifo_map_space,
			    mn->a24_pagestart + mn->a24_fifo_pagestart,
			    FIFO_PAGE_NUM, NULL);
	if (mn->a24_fifo_base == NULL)
	{
	    i1432_error_info = igeterrstr(igeterrno());
	    return i1432_la_print_error(mn->la, ERR1432_UNABLE_TO_MAP_MEMORY);
	}
    }

    /* Save the imap count, for use when unmapping */
    mn->a24_fifo_imap_count = i1432_sicl_imap_count++;

    return 0;
}

static SHORTSIZ16
sicl_iunmap_normal(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error;

    if (mn->a24_base != NULL)
    {
	error = sicl_iunmap_down_to(mn->a24_imap_count + 1);
	if (error)
	    return error;

	/* Don't try to include this in the call to
	   sicl_iunmap_down_to, because it will fail if mn is a
	   separate module-list-node that is not part of the
	   i1432_mod_list array.  This is the case during
	   e1432_get_hwconfig and e1432_install. */
	error = sicl_iunmap_normal_help(mn);
	if (error)
	    return error;
    }

    return 0;
}

static SHORTSIZ16
sicl_iunmap_fifo(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error;

    if (mn->a24_fifo_base != NULL)
    {
	error = sicl_iunmap_down_to(mn->a24_fifo_imap_count + 1);
	if (error)
	    return error;

	error = sicl_iunmap_fifo_help(mn);
	if (error)
	    return error;
    }

    return 0;
}

SHORTSIZ16
i1432_sicl_imap(E1432_MODULE_LIST_NODE *mn, int use_fifo)
{
    if (use_fifo)
	return sicl_imap_fifo(mn);
    else
	return sicl_imap_normal(mn);
}

SHORTSIZ16
i1432_sicl_iunmap(E1432_MODULE_LIST_NODE *mn, int use_fifo)
{
    if (use_fifo)
	return sicl_iunmap_fifo(mn);
    else
	return sicl_iunmap_normal(mn);
}

#endif	/* HAVE_SICL */
